#if !defined( _STDDEFS_FXH )
#define _STDDEFS_FXH

//---------------------------------------------------------------------------------
//
// HLSL header file contains common macros and definitions.
//

#ifdef _XBOX
#define	REMOVE_UNUSED_INTERPOLATORS		[removeUnusedInputs]
#define INSTANCES_CACHE					UseTextureCache=false
#else
#define	REMOVE_UNUSED_INTERPOLATORS
#endif

// USE_WVP_CONSTANT is defined when we want worlviewproj passed as a constant, rather than as separate world and viewproj matrices
#if defined(_PS3_)
#define USE_WVP_CONSTANT
#elif defined(_XBOX)
#define USE_WVP_CONSTANT
#elif !defined(_3DSMAX_)
#define USE_WVP_CONSTANT
#endif

//
// Shadow Defines
//
#ifdef VSM_SHADOWGEN
	#define	CALC_SHADOWMAP_DEPTH( _out, _coords_in )	{ _out.r = _coords_in.z/_coords_in.w; _out.g = _out.r*_out.r; _out.ba = 0.0f; }
#else

	#ifdef _XBOX
		//#define SHADOWMAP_COLOUR	// we're rendering the shadow map to a colour buffer
		//#define SHADOWMAP_LINEAR_Z	// we're using linear Z rather than z/w
	#endif
	
	#ifdef _PS3_
		#define CALC_SHADOWMAP_DEPTH( _out, _coords_in )	// Again does nothing, but there seem to be alot of defines for it in code
	#else
		#ifdef SHADOWMAP_LINEAR_Z
			#define	CALC_SHADOWMAP_DEPTH( _out, _coords_in )	{ _out = _coords_in.z; }
		#else
			#define	CALC_SHADOWMAP_DEPTH( _out, _coords_in )	{ _out = _coords_in.z/_coords_in.w; }
		#endif
	#endif
#endif

#if (!defined(_XBOX) && !defined(_PS3_)) || defined(SHADOWMAP_COLOUR)
#define NEEDS_SHADOW_COORDS			// defined if we need to output the shadow coords to the colour buffer
#endif

#if !defined(_PS3_)
#define	OUTPUT_SHADOW_COORDS	_output.shadowCoord = _output.position;
#else
#define	OUTPUT_SHADOW_COORDS	// Not required on PS3 (But sometimes the above isn't defined)
#endif

#if defined(_PS3_)
#define	CLAMP_SHADOW_Z				_output.position.z=max(-1.0f, _output.position.z);
#else
#define	CLAMP_SHADOW_Z				_output.position.z=max(0.0f, _output.position.z);
#endif

#ifdef SHADOWGEN_USE_CLIP

	// if Alpha Test is not available for rendering to the shadows, we have to use clip instead
	#define	SHADOWMAP_PS_ALPHATEST( _alpha, _value )			clip( (_alpha)-(_value) )
	#define SHADOWMAP_STATES_ALPHATEST( _value )					AlphaTestEnable = false; AlphaBlendEnable = false;
	
#else

	// Alpha Test is available for rendering to the shadows, so use the render states
	#define	SHADOWMAP_PS_ALPHATEST( _alpha, _value )			
	#if defined(_PS3_)
		#define SHADOWMAP_STATES_ALPHATEST( _value )	\
			AlphaTestEnable = true;\
			AlphaFunc = int2(GEqual, _value);
	#elif defined(_XBOX)
		#define SHADOWMAP_STATES_ALPHATEST( _value )	\
			AlphaTestEnable = true;\
			AlphaRef = _value;\
		    AlphaFunc = GreaterEqual;
	#else
		#define SHADOWMAP_STATES_ALPHATEST( _value )	\
			AlphaBlendEnable = false;\
			AlphaTestEnable = true;\
			AlphaRef = _value;\
		  AlphaFunc = GreaterEqual;
	#endif
	
#endif


// 3dsMax needs this string in order to choose which parser to apply.
#ifdef _3DSMAX_

#define DCC_MAX

string ParamID = "0x0000";
/*
float Script : STANDARDSGLOBAL <
	string UIWidget = "none";
	string ScriptClass = "scene";
	string ScriptOrder = "standard";
	string ScriptOutput = "color";
	string Script = "Technique=Basic;";
> = 0.8; // version #
*/
#endif

//
// Mipmap filtering uses a different name in CG
//
#if defined(_PS3_)
#define _MINFILTER	LinearMipMapLinear
#define _ANISOMINFILTER	LinearMipMapLinear
#define _ANISOMAXFILTER	Linear
#else
#define _MINFILTER	Linear
#define _ANISOMINFILTER	Anisotropic
#define _ANISOMAXFILTER	Anisotropic
#endif

//
// Half for use in lighting calcs
//
// LPFLOAT - Low precesion float
//

#if defined(_PS3_)
#define LPFLOAT		float
#define LPFLOAT2	float2
#define LPFLOAT3	float3
#define LPFLOAT4	float4
#define Rv(X)		X##f
#else
#define LPFLOAT		float
#define LPFLOAT2	float2
#define LPFLOAT3	float3
#define LPFLOAT4	float4
#define Rv(X)		X##f
#endif


// Max doesn't care about anisotropy
#if defined( _3DSMAX_)

#define SET_MAX_ANISOTROPY( level )
#define SET_NO_ANISOTROPY

#else

// Max anisotropy setting (keep it consistent across shaders, different syntax on Ps3)
#if defined(_PS3_)
#define SET_MAX_ANISOTROPY( level )	PS3MaxAnisotropy = Aniso##level;
#define SET_NO_ANISOTROPY PS3MaxAnisotropy = Aniso1;
#else
#define SET_MAX_ANISOTROPY( level )	MaxAnisotropy = level;
#define SET_NO_ANISOTROPY MaxAnisotropy = 1;
#endif

#endif


#if defined(_XBOX)
//#define	SHARE_PARAM		shared
#define SHARE_PARAM
#else
#define SHARE_PARAM
#endif

#if defined(_PS3_) || defined(_3DSMAX_)
#define	SET_FX_SAMPLER_STATES
#endif


//
// Output colour type (needs to be half on the PS3 so that SRGB writes will work)
//

#if defined( _PS3_ )

#define COLOUR_OUTPUT_TYPE half4

#else

#define COLOUR_OUTPUT_TYPE float4

#endif




//
// Gamma correction sampler states
//

#define SET_SRGB_TEXTURE									int SRGBTexture = 1;
#define SET_LINEAR_TEXTURE								int SRGBTexture = 0;

// The xb360 doesn't support the SRGBTexture sampler state
#if defined( _XBOX )

#define FX_SAMPLERSTATE_SRGB_TEXTURE
#define FX_SAMPLERSTATE_LINEAR_TEXTURE

#else

#define FX_SAMPLERSTATE_SRGB_TEXTURE			SRGBTexture = 1;
#define FX_SAMPLERSTATE_LINEAR_TEXTURE		SRGBTexture = 0;

#endif



//-----------------------------------------------------------------------
//
// Common Functions
//


//
// Converts from linear colour space to sRGB
//
float4 LinearTosRGB( float4 _linearCol )
{
#if defined( _PS3_ )

	// PS3 sRGB encoding
	return float4( pow( _linearCol.rgb, 1.0f / 2.2f ), _linearCol.a );

#elif defined( _XBOX )

	// XBOX sRGB encoding
	return float4( pow( _linearCol.rgb, 1.0f / 2.2f ), _linearCol.a );

#else

	// PC sRGB encoding
	return float4( pow( _linearCol.rgb, 1.0f / 2.2f ), _linearCol.a );

#endif	// defined( _PS3_ )
}


//
// Converts from sRGB colour space to linear
//
float4 sRGBToLinear( float4 _sRGBCol )
{
#if defined( _PS3_ )

	// PS3 sRGB decoding
	return float4( pow( _sRGBCol.rgb, 2.2f ), _sRGBCol.a );

#elif defined( _XBOX )

	// XBOX sRGB encoding
	return float4( pow( _sRGBCol.rgb, 2.2f ), _sRGBCol.a );

#else

	// PC sRGB decoding
	return float4( pow( _sRGBCol.rgb, 2.2f ), _sRGBCol.a );

#endif	//  defined( _PS3_ )
}






//-----------------------------------------------------------------------
//
// CalcTangentSpaceNormalMapping - Function performs tangent
//				space normal mapping given an input tangent space coord frame
//				and normal map input vector.
//

float3 CalcTangentSpaceNormalMapping( LPFLOAT3 _normal, LPFLOAT3 _binormal, LPFLOAT3 _tangent, LPFLOAT3 _mapNormal )
{
#if defined( _3DSMAX_ )
	// 3ds max swaps y and z
	return normalize( ( _mapNormal.z * _normal ) + ( _mapNormal.x * _binormal ) + ( _mapNormal.y * _tangent ) );
#else
	return normalize( ( _mapNormal.z * _normal ) + ( _mapNormal.x * _binormal ) + ( _mapNormal.y * _tangent ) );
#endif
}


//-----------------------------------------------------------------------
//
// CalculateSafeReflection - Function calculates a reflection vector from an
//																		incident vector and normal. A variation on the
//																		standard formula is used to ensure the reflected
//																		vector is infront of the normal.
//

LPFLOAT3 CalculateSafeReflection( LPFLOAT3 _I, LPFLOAT3 _N )
{
	return _I + ( Rv( 2.0 ) * abs( dot( _I, _N ) ) * _N );
}


//-----------------------------------------------------------------------
//
// CalculateH - Function calculates the half-vector H for use in Blinn-style
//							lighting calculations.
//

LPFLOAT3 CalculateH( LPFLOAT3 _lightVec, LPFLOAT3 _eyeVec )
{
	return normalize( _lightVec + _eyeVec );
}


//-----------------------------------------------------------------------
//
// CalculateDiffuseLighting - Function calculates the Lambertian diffuse lighting
//																		level for a given normal and light vector.
//

LPFLOAT CalculateDiffuseLighting( LPFLOAT3 _N, LPFLOAT3 _L )
{
	return dot( _N, _L );
}



//-----------------------------------------------------------------------
//
// CalculateHalfLambertDiffuseLighting - Function calculates the diffuse lighting
//																										level for a given normal and light vector,
//																										using a half-Lambertian approach,
//																										which wraps around to the back of objects.
//

LPFLOAT CalculateHalfLambertDiffuseLighting( LPFLOAT3 _N, LPFLOAT3 _L )
{
	return ( dot( _N, _L ) + 0.5f ) * 0.75f;
}



//-----------------------------------------------------------------------
//
// CalculateSpecularLighting - Function calculates the specular lighting level
//														 for a given normal and light vector, using the
//														 Blinn formula.
//

LPFLOAT CalculateSpecularLighting( LPFLOAT3 _N, LPFLOAT3 _L, LPFLOAT3 _E, LPFLOAT _pow )
{
	// Calculate H
	LPFLOAT3 H = CalculateH( _L, _E );

	// Calculate lighting level
	return pow( max ( dot( _N, H ), 0.0000001f ), _pow );
}



//-----------------------------------------------------------------------
//
// CalculateSpecularLighting - Function calculates the specular lighting level
//														 using the anisotropic specular formula from
//														 [WARD92].
//

LPFLOAT CalculateWardSpecularLighting( LPFLOAT3 _N, LPFLOAT3 _X, LPFLOAT3 _Y, LPFLOAT _aX, LPFLOAT _aY, LPFLOAT3 _L, LPFLOAT3 _E )
{
	// Calculate H
	LPFLOAT3 H = CalculateH( _L, _E );

	// Calculate the anisotropy level from the input values, using the ward equation
	LPFLOAT a = Rv(1.0) / sqrt( dot( _N, _L ) * dot( _N, _E ) );
	LPFLOAT b = dot( _N, _L ) / ( Rv(4.0) * Rv(3.14159265) * _aX * _aY );

	LPFLOAT denominator = Rv(1.0) + dot( H, _N );
	LPFLOAT Xfactor = dot( H, _X ) / _aX;
	LPFLOAT Yfactor = dot( H, _Y ) / _aY;

	LPFLOAT exponent = Rv(-2.0) * ( ( Xfactor * Xfactor ) + ( Yfactor * Yfactor ) ) / denominator;
	LPFLOAT c = exp( exponent );

	return ( a * b * c );

}



//-----------------------------------------------------------------------
//
// CalculateSpotLightAttenuation - Calculates the attenuation factor for a
//																 spot lightsource at a given point.
//
// Parameters:
//	_pointToLight : Vector from the point to the light source.
//	_dir					: Light direction
//	_attFactor		: Attenuation factors ( = 1.0 / ( end dist squared )
//	_angles				: Cosines of the inner (x) and outer (y) spotlight angles.
//
// Return:
//	The attenuation factor for the light source at this position, [0.0 ... 1.0].
//
//-----------------------------------------------------------------------

LPFLOAT CalculateSpotLightAttenuation( LPFLOAT3 _pointToLight, LPFLOAT3 _dir, LPFLOAT _attFactor, LPFLOAT2 _angles )
{
	LPFLOAT att;

	// Calculate distance attenuation
	LPFLOAT distSquared = dot( _pointToLight, _pointToLight );
	att = smoothstep( Rv(0.0), Rv(1.0), Rv(1.0) - ( distSquared * _attFactor ) );

	// Calculate angle attenuation
	LPFLOAT cosAngle = - dot( _dir, _pointToLight / sqrt( distSquared ) );
	LPFLOAT angleAttenuation = smoothstep( _angles.y, _angles.x, cosAngle );

	// Combine the two
	att *= angleAttenuation;

	return att;
}


//-----------------------------------------------------------------------
//
// CalculatePointLightAttenuation - Calculates the attenuation factor for a point
//																	lightsource at a given point.
//
// Parameters:
//	_pointToLight : Vector from the point to the light source.
//	_attFactor		: Attenuation factors ( = 1.0 / ( end dist squared )
//
// Return:
//	The attenuation factor for the light source at this position, [0.0 ... 1.0].
//
//-----------------------------------------------------------------------

LPFLOAT CalculatePointLightAttenuation( LPFLOAT3 _pointToLight, LPFLOAT _attFactor )
{
	LPFLOAT att;

	// Calculate distance attenuation
	LPFLOAT distSquared = dot( _pointToLight, _pointToLight );
	att = smoothstep( Rv(0.0), Rv(1.0), Rv(1.0) - ( distSquared * _attFactor ) );

	return att;
}

//-----------------------------------------------------------------------
//
// CalculateOutputPixel - Calculates the output pixel HDR for PS3
// ignored by PC/360
//
//-----------------------------------------------------------------------

// Constants for PS3
#ifdef _PS3_

#define USE_HDR_TEXTURE_METHOD		// Use new method with a texture lut

#ifdef USE_HDR_TEXTURE_METHOD
texture HDRTexture : TEXTURE < bool appEdit = false; >;
sampler2D HDRTextureSampler : SAMPLER < bool appEdit = false; string SamplerTexture="HDRTexture"; > = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	MinFilter = LinearMipMapLinear;
	MagFilter = Linear;
	MipFilter = None;
	SET_NO_ANISOTROPY
};
float4 HDRParams;
#else
float4 HDRMode;
float4 HDRValues;
#endif
#endif

#ifdef USE_HDR_TEXTURE_METHOD
half4 CalculateOutputPixel(float4 colour)
{
#ifdef _PS3_
	float lLuminance = dot( float4(colour.rgb, 1.0f), HDRParams );
		
	half2 lut=tex2D(HDRTextureSampler, float2(lLuminance, 0.25f+0.5f*colour.a));
		
	return half4(colour.rgb*lut.x, lut.y);
	
#elif defined( _3DSMAX_ )

	// Gamma correct the result, as max doesn't do it
	colour.rgb = pow( colour.rgb, 1.0f / 2.2f );
	return colour;

#else

	return colour;

#endif
}
#else
half4 CalculateOutputPixel(half4 colour)
{
#ifdef _PS3_
	if (HDRMode.x)
	{
		half lLuminance = dot( colour.rgb, half3(0.333h,0.333h,0.333h) );
		
		half Lp=HDRValues.x*lLuminance;
		half toneScalar=(Lp*(1.0h+(Lp/(HDRValues.y))))/(1.0h+Lp);
		colour.rgb*=lerp(1.0h, toneScalar, HDRValues.w);
		if (HDRMode.y)
		{
			colour.a=lLuminance*HDRValues.z;
		}
	}
	
	return colour;
		
#elif defined( _3DSMAX_ )

	// Gamma correct the result, as max doesn't do it
	colour.rgb = pow( abs(colour.rgb), 1.0f / 2.2f );
	return colour;

#else

	return colour;

#endif
}
#endif

half4 CalculateLowDetailOutputPixel(half4 colour)
{
#if defined( _3DSMAX_ )

	// Gamma correct the result, as max doesn't do it
	colour.rgb = pow( abs(colour.rgb), 1.0f / 2.2f );
	return colour;

#else

	return colour;

#endif
}

//
// Readability defines for parameter coherence types
// Need to match the enum CShader::ECoherenceType
//
#define ECT_DYNAMIC 1
#define ECT_WITHIN_MATERIAL 2
#define ECT_WITHIN_SHADER 3
#define ECT_WITHIN_SCENE 4



//
// Preprocessor cunningness to generate texcoord semantics from two
// input values (base + offset).
//

#define TEXCOORDDEF( x, y ) TEXCOORD_GEN_ ## x ## y

#define TEXCOORD_GEN_00 TEXCOORD0;
#define TEXCOORD_GEN_01 TEXCOORD1;
#define TEXCOORD_GEN_02 TEXCOORD2;
#define TEXCOORD_GEN_03 TEXCOORD3;
#define TEXCOORD_GEN_04 TEXCOORD4;
#define TEXCOORD_GEN_05 TEXCOORD5;
#define TEXCOORD_GEN_06 TEXCOORD6;
#define TEXCOORD_GEN_07 TEXCOORD7;
#define TEXCOORD_GEN_08 TEXCOORD8;
#define TEXCOORD_GEN_09 TEXCOORD9;

#define TEXCOORD_GEN_10 TEXCOORD1;
#define TEXCOORD_GEN_11 TEXCOORD2;
#define TEXCOORD_GEN_12 TEXCOORD3;
#define TEXCOORD_GEN_13 TEXCOORD4;
#define TEXCOORD_GEN_14 TEXCOORD5;
#define TEXCOORD_GEN_15 TEXCOORD6;
#define TEXCOORD_GEN_16 TEXCOORD7;
#define TEXCOORD_GEN_17 TEXCOORD8;
#define TEXCOORD_GEN_18 TEXCOORD9;

#define TEXCOORD_GEN_20 TEXCOORD2;
#define TEXCOORD_GEN_21 TEXCOORD3;
#define TEXCOORD_GEN_22 TEXCOORD4;
#define TEXCOORD_GEN_23 TEXCOORD5;
#define TEXCOORD_GEN_24 TEXCOORD6;
#define TEXCOORD_GEN_25 TEXCOORD7;
#define TEXCOORD_GEN_26 TEXCOORD8;
#define TEXCOORD_GEN_27 TEXCOORD9;

#define TEXCOORD_GEN_30 TEXCOORD3;
#define TEXCOORD_GEN_31 TEXCOORD4;
#define TEXCOORD_GEN_32 TEXCOORD5;
#define TEXCOORD_GEN_33 TEXCOORD6;
#define TEXCOORD_GEN_34 TEXCOORD7;
#define TEXCOORD_GEN_35 TEXCOORD8;
#define TEXCOORD_GEN_36 TEXCOORD9;

#define TEXCOORD_GEN_40 TEXCOORD4;
#define TEXCOORD_GEN_41 TEXCOORD5;
#define TEXCOORD_GEN_42 TEXCOORD6;
#define TEXCOORD_GEN_43 TEXCOORD7;
#define TEXCOORD_GEN_44 TEXCOORD8;
#define TEXCOORD_GEN_45 TEXCOORD9;

#define TEXCOORD_GEN_50 TEXCOORD5;
#define TEXCOORD_GEN_51 TEXCOORD6;
#define TEXCOORD_GEN_52 TEXCOORD7;
#define TEXCOORD_GEN_53 TEXCOORD8;
#define TEXCOORD_GEN_54 TEXCOORD9;

#define TEXCOORD_GEN_60 TEXCOORD6;
#define TEXCOORD_GEN_61 TEXCOORD7;
#define TEXCOORD_GEN_62 TEXCOORD8;
#define TEXCOORD_GEN_63 TEXCOORD9;

#define TEXCOORD_GEN_70 TEXCOORD7;
#define TEXCOORD_GEN_71 TEXCOORD8;
#define TEXCOORD_GEN_72 TEXCOORD9;

#define TEXCOORD_GEN_80 TEXCOORD8;
#define TEXCOORD_GEN_81 TEXCOORD9;

#define TEXCOORD_GEN_90 TEXCOORD9;


#endif	//  _STDDEFS_FXH


//Compiler tweaks
#define D3DXSHADER_OPTIMIZATION_LEVEL3;
#define D3DXSHADER_OPTIMIZATION_LEVEL 3;
